using System;
using System.Runtime.InteropServices;
using System.Threading;
using NDI.SLIKDA.Interop;
using CSWindowsService;


namespace CSWindowsService
{
	public class CSimulate
	{
		private bool m_bRun = true;
		private SLIKServer m_Svr = null;
		private Thread m_SimulatorThread = null;

		public CSimulate( SLIKServer Svr )
		{
			m_Svr = Svr;		
		}

		public void StartSimulation()
		{
			m_SimulatorThread = new Thread(new ThreadStart(Simulate));
			m_SimulatorThread.ApartmentState = ApartmentState.MTA;
			m_SimulatorThread.Start();
		}

		public void StopSimulation()
		{
			m_bRun = false;
			m_SimulatorThread.Join();
		}

		//	The method used to simulate data changes,
		//	in a separate thread.  This may be typical of a 
		//	"real" server that is interacting with an underlying
		//	device / system to update the tags in the OPC
		//	server data cache (i.e. the SLIKTags collection).
		[MTAThread]
		private void Simulate()
		{
			ISLIKTag tag = m_Svr.SLIKTags[1];
			while( m_bRun )
			{
				UpdateTags( m_Svr.SLIKTags );
				Thread.Sleep( 1000 );	
			}
		}
		
		private void UpdateTags( 
			ISLIKTags Tags 
			)
		{
			//
			// A real server might scan one or more devices here to update the Tags.
			// This server does not communicate with any hardware, so instead we will
			// simulate data changes.
			//
   
			// Update each SLIKTag in the SLIKTags collection.  Note that SLIKTags
			// collection indices are 1-based!
			for( int i = 1; i <= Tags.Count; i++ )
			{
				//
				// Simulate data changes for each tag that meets the following criteria:
				//   1. The tag is currently active.
				//   2. The tag's access permessions are read-only.
				//
				ISLIKTag Tag = Tags[i];
				if ( Tag.Active && ( Tag.AccessPermissions == (int) AccessPermissionsEnum.sdaReadAccess ) )
					SimulateDataChange( Tag, (int) QualityStatusEnum.sdaGood );
				else
					// Update the quality (and timestamp) for the other tags
					Tag.SetVQT( DefaultValues.SetVQT_Value, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.SetVQT_Timestamp 
						);
			}
		}

		private static void SimulateDataChange(
			ISLIKTag	Tag,
			int			nQuality
			)
		{    
			object Value = Tag.Value;
			switch( (VarEnum) Tag.DataType )
			{
				case VarEnum.VT_BOOL:
					bool bVal = (bool) Value;
					Value = !bVal;
					break;

				case VarEnum.VT_UI1:
					byte byVal = (byte) Value;
					if( byVal > 250 )
						byVal = 0;
					else
						++byVal;
					Value = byVal;
					break;

				case VarEnum.VT_I2:
					short shVal = (short) Value;
					if( shVal > 10000 )
						shVal = -10000;
					else
						shVal += 4;
					Value = shVal;
					break;

				case VarEnum.VT_I4:
					int nVal = (int) Value;
					if( nVal > 1000000 ) 
						nVal = -1000000;
					else
						nVal += 9;
					Value = nVal;
					break;

				case VarEnum.VT_R4:
					float fltVal = (float) Value;
					if( fltVal > 1E+20 )
						fltVal = -1E+20f;
					if( fltVal > 0 )
						fltVal *= 1.1f;
					else
						fltVal = (fltVal / 1.1f) + 1f;
					Value = fltVal;
					break;

				case VarEnum.VT_R8:
					double dblVal = (double) Value;
					if( dblVal > 1E+50 )
						dblVal = -1E+50;
					if( dblVal > 0 )
						dblVal *= 1.2F;
					else
						dblVal = (dblVal / 1.2F) + 1F;
					Value = dblVal;
					break;

				case VarEnum.VT_CY:
					Value = new CurrencyWrapper( (decimal) Value + 1 );
					break;

				case VarEnum.VT_DATE:
					Value = System.DateTime.Now;
					break;

				case VarEnum.VT_BSTR:
					Value = "Current time is " + System.DateTime.Now.ToLongTimeString();
					break;

				default:
					// Simulation for data type not supported
					break;
			}

			// Now set the Value, Quality, and Timestamp
			// Note: By passing the default value for the timestamp to SetVQT(),
			//       SLIK-DA will use the current system time,
			//       and generate a timestamp with millisecond resolution.
			Tag.SetVQT( 
				Value, 
				(short) nQuality, 
				DefaultValues.SetVQT_Timestamp 
				);
		}
	}
}


	
	
	
